#!/bin/sh
# setup.sh
# This script should be run after the webmin archive is unpacked, in order
# to setup the various config files

# Find install directory
cd `dirname $0`
if [ -x /bin/pwd ]; then
	wadir=`/bin/pwd`
else
	wadir=`pwd`;
fi
ver=`cat $wadir/version`

echo "***********************************************************************"
echo "*            Welcome to the Webmin setup script, version $ver         *"
echo "***********************************************************************"
echo "Webmin is a web-based interface that allows Unix-like operating"
echo "systems and common Unix services to be easily administered."
echo ""

# Only root can run this
id | grep "uid=0(" >/dev/null
if [ $? != "0" ]; then
	echo "ERROR: The Webmin install script must be run as root";
	echo "";
	exit 1;
fi

# Display install directory
echo "Installing Webmin in $wadir ..."
allmods=`cd $wadir; ls */module.info | sed -e 's/\/module.info//g' | xargs echo`
echo ""

# Ask for webmin config directory
echo "***********************************************************************"
echo "Webmin uses separate directories for configuration files and log files."
echo "Unless you want to run multiple versions of Webmin at the same time"
echo "you can just accept the defaults."
echo ""
printf "Config file directory [/etc/webmin]: "
if [ "$config_dir" = "" ]; then
	read config_dir
fi
if [ "$config_dir" = "" ]; then
	config_dir=/etc/webmin
fi
abspath=`echo $config_dir | grep "^/"`
if [ "$abspath" = "" ]; then
	echo "Config directory must be an absolute path"
	echo ""
	exit 2
fi
if [ ! -d $config_dir ]; then
	mkdir $config_dir;
	if [ $? != 0 ]; then
		echo "ERROR: Failed to create directory $config_dir"
		echo ""
		exit 2
	fi
fi
if [ -r "$config_dir/config" ]; then
	echo "Found existing Webmin configuration in $config_dir"
	echo ""
	upgrading=1
fi

# Check if upgrading from an old version
if [ "$upgrading" = 1 ]; then
	# Get current var path
	var_dir=`cat $config_dir/var-path`

	# Get current perl path
	perl=`cat $config_dir/perl-path`

	# Stop old version
	$config_dir/stop >/dev/null 2>&1

	# Update ACLs
	$perl $wadir/newmods.pl $config_dir $allmods

	# Get old os name and version
	os_type=`grep "^os_type=" $config_dir/config | sed -e 's/os_type=//g'`
	os_version=`grep "^os_version=" $config_dir/config | sed -e 's/os_version=//g'`
	real_os_type=`grep "^real_os_type=" $config_dir/config | sed -e 's/real_os_type=//g'`
	real_os_version=`grep "^real_os_version=" $config_dir/config | sed -e 's/real_os_version=//g'`

	# Get old root, host, port, ssl and boot flag
	oldwadir=`grep "^root=" $config_dir/miniserv.conf | sed -e 's/root=//g'`
	host=`grep "^host=" $config_dir/miniserv.conf | sed -e 's/host=//g'`
	port=`grep "^port=" $config_dir/miniserv.conf | sed -e 's/port=//g'`
	ssl=`grep "^ssl=" $config_dir/miniserv.conf | sed -e 's/ssl=//g'`
	atboot=`grep "^atboot=" $config_dir/miniserv.conf | sed -e 's/atboot=//g'`

	# Update miniserv.conf with new root directory and mime types file
	grep -v "^root=" $config_dir/miniserv.conf | grep -v "^mimetypes=" >/tmp/$$.miniserv.conf
	mv /tmp/$$.miniserv.conf $config_dir/miniserv.conf
	echo "root=$wadir" >> $config_dir/miniserv.conf
	echo "mimetypes=$wadir/mime.types" >> $config_dir/miniserv.conf
	grep logout= $config_dir/miniserv.conf >/dev/null
	if [ $? != "0" ]; then
		echo "logout=$config_dir/logout-flag" >> $config_dir/miniserv.conf
	fi
	
	# Check for third-party modules in old version
	if [ "$wadir" != "$oldwadir" ]; then
		echo "Checking for third-party modules .."
		if [ "$webmin_upgrade" != "" ]; then
			autothird=1
		fi
		$perl $wadir/thirdparty.pl $wadir $oldwadir $autothird
		echo "..done"
		echo ""
	fi
else
	# Config directory exists .. make sure it is not in use
	ls $config_dir | grep -v rpmsave >/dev/null 2>&1
	if [ "$?" = "0" -a "$config_dir" != "/etc/webmin" ]; then
		echo "ERROR: Config directory $config_dir is not empty"
		echo ""
		exit 2
	fi

	# Ask for log directory
	printf "Log file directory [/var/webmin]: "
	if [ "$var_dir" = "" ]; then
		read var_dir
	fi
	if [ "$var_dir" = "" ]; then
		var_dir=/var/webmin
	fi
	abspath=`echo $var_dir | grep "^/"`
	if [ "$abspath" = "" ]; then
		echo "Log file directory must be an absolute path"
		echo ""
		exit 3
	fi
	if [ ! -d $var_dir ]; then
		mkdir $var_dir
		if [ $? != 0 ]; then
			echo "ERROR: Failed to create directory $var_dir"
			echo ""
			exit 3
		fi
	fi
	echo ""

	# Ask where perl is installed
	echo "***********************************************************************"
	echo "Webmin is written entirely in Perl. Please enter the full path to the"
	echo "Perl 5 interpreter on your system."
	echo ""
	if [ -x /usr/bin/perl ]; then
		perldef=/usr/bin/perl
	elif [ -x /usr/local/bin/perl ]; then
		perldef=/usr/local/bin/perl
	else
		perldef=""
	fi
	if [ "$perl" = "" ]; then
		if [ "$perldef" = "" ]; then
			printf "Full path to perl: "
			read perl
			if [ "$perl" = "" ]; then
				echo "ERROR: No path entered!"
				echo ""
				exit 4
			fi
		else
			printf "Full path to perl (default $perldef): "
			read perl
			if [ "$perl" = "" ]; then
				perl=$perldef
			fi
		fi
	fi
	echo ""

	# Test perl 
	echo "Testing Perl ..."
	if [ ! -x $perl ]; then
		echo "ERROR: Failed to find perl at $perl"
		echo ""
		exit 5
	fi
	$perl -e 'print "foobar\n"' 2>/dev/null | grep foobar >/dev/null
	if [ $? != "0" ]; then
		echo "ERROR: Failed to run test perl script. Maybe $perl is"
		echo "not the perl interpreter, or is not installed properly"
		echo ""
		exit 6
	fi
	$perl -e 'exit ($] < 5.002 ? 1 : 0)'
	if [ $? = "1" ]; then
		echo "ERROR: Detected old perl version. Webmin requires"
		echo "perl 5.002 or better to run"
		echo ""
		exit 7
	fi
	$perl -e 'use Socket; print "foobar\n"' 2>/dev/null | grep foobar >/dev/null
	if [ $? != "0" ]; then
		echo "ERROR: Perl Socket module not installed. Maybe Perl has"
		echo "not been properly installed on your system"
		echo ""
		exit 8
	fi
	$perl -e '$c = crypt("xx", "yy"); exit($c ? 0 : 1)'
	if [ $? != "0" ]; then
		echo "ERROR: Perl crypt function does not work. Maybe Perl has"
		echo "not been properly installed on your system"
		echo ""
		exit 8
	fi
	echo "Perl seems to be installed ok"
	echo ""

	# Ask for operating system type
	echo "***********************************************************************"
	if [ "$os_type" = "" ]; then
		$perl $wadir/oschooser.pl $wadir/os_list.txt /tmp/$$.os $autoos
		if [ $? != 0 ]; then
			exit $?
		fi
		. /tmp/$$.os
		rm -f /tmp/$$.os
	fi
	echo "Operating system name:    $real_os_type"
	echo "Operating system version: $real_os_version"
	echo ""

	# Ask for web server port, name and password
	echo "***********************************************************************"
	echo "Webmin uses its own password protected web server to provide access"
	echo "to the administration programs. The setup script needs to know :"
	echo " - What port to run the web server on. There must not be another"
	echo "   web server already using this port."
	echo " - The login name required to access the web server."
	echo " - The password required to access the web server."
	echo " - The hostname of this system that the web server should use."
	echo " - If the webserver should use SSL (if your system supports it)."
	echo " - Whether to start webmin at boot time."
	echo ""
	printf "Web server port (default 10000): "
	if [ "$port" = "" ]; then
		read port
		if [ "$port" = "" ]; then
			port=10000
		fi
	fi
	if [ $port -lt 1 ]; then
		echo "ERROR: $port is not a valid port number"
		echo ""
		exit 11
	fi
	if [ $port -gt 65535 ]; then
		echo "ERROR: $port is not a valid port number. Port numbers cannot be"
		echo "       greater than 65535"
		echo ""
		exit 12
	fi
	$perl -e 'use Socket; socket(FOO, PF_INET, SOCK_STREAM, getprotobyname("tcp")); setsockopt(FOO, SOL_SOCKET, SO_REUSEADDR, pack("l", 1)); bind(FOO, sockaddr_in($ARGV[0], INADDR_ANY)) || exit(1); exit(0);' $port
	if [ $? != "0" ]; then
		echo "ERROR: TCP port $port is already in use by another program"
		echo ""
		exit 13
	fi
	printf "Login name (default admin): "
	if [ "$login" = "" ]; then
		read login
		if [ "$login" = "" ]; then
			login="admin"
		fi
	fi
	echo $login | grep : >/dev/null
	if [ "$?" = "0" ]; then
		echo "ERROR: Username contains a : character"
		echo ""
		exit 14
	fi
	printf "Login password: "
	if [ "$password" = "" -a "$crypt" = "" ]; then
		stty -echo
		read password
		stty echo
		printf "\n"
		printf "Password again: "
		stty -echo
		read password2
		stty echo
		printf "\n"
		if [ "$password" != "$password2" ]; then
			echo "ERROR: Passwords don't match"
			echo ""
			exit 14
		fi
		echo $password | grep : >/dev/null
		if [ "$?" = "0" ]; then
			echo "ERROR: Password contains a : character"
			echo ""
			exit 14
		fi
	fi
	defhost=`hostname`
	printf "Web server hostname (default $defhost): "
	if [ "$host" = "" ]; then
		read host
		if [ "$host" = "" ]; then
			host=$defhost
		fi
	fi
	if [ "$ssl" = "" ]; then
		ssl=0
		$perl -e 'use Net::SSLeay' >/dev/null 2>/dev/null
		if [ $? = "0" ]; then
			printf "Use SSL (y/n): "
			read sslyn
			if [ "$sslyn" = "y" -o "$sslyn" = "Y" ]; then
				ssl=1
			fi
		else
			echo "The Perl SSLeay library is not installed. SSL not available."
			rm -f core
		fi
	fi

	# Ask whether to run at boot time
	if [ "$atboot" = "" ]; then
		initsupp=`grep "^os_support=" $wadir/init/module.info | sed -e 's/os_support=//g' | grep $os_type`
		atboot=0
		if [ "$initsupp" != "" ]; then
			printf "Start Webmin at boot time (y/n): "
			read atbootyn
			if [ "$atbootyn" = "y" -o "$atbootyn" = "Y" ]; then
				atboot=1
			fi
		else
			echo "Webmin does not support being started at boot time on your system."
		fi
	fi
	makeboot=$atboot

	# Create webserver config file
	echo $perl > $config_dir/perl-path
	echo $var_dir > $config_dir/var-path
	echo "***********************************************************************"
	echo "Creating web server config files.."
	cfile=$config_dir/miniserv.conf
	echo "port=$port" >> $cfile
	echo "root=$wadir" >> $cfile
	echo "host=$host" >> $cfile
	echo "mimetypes=$wadir/mime.types" >> $cfile
	echo "addtype_cgi=internal/cgi" >> $cfile
	echo "realm=Webmin Server" >> $cfile
	echo "logfile=$var_dir/miniserv.log" >> $cfile
	echo "pidfile=$var_dir/miniserv.pid" >> $cfile
	echo "logtime=168" >> $cfile
	echo "ppath=$ppath" >> $cfile
	echo "ssl=$ssl" >> $cfile
	echo "env_WEBMIN_CONFIG=$config_dir" >> $cfile
	echo "env_WEBMIN_VAR=$var_dir" >> $cfile
	echo "atboot=$atboot" >> $cfile
	echo "logout=$config_dir/logout-flag" >> $cfile
	echo "listen=10000" >> $cfile
	echo "denyfile=\\.pl\$" >> $cfile
	if [ "$allow" != "" ]; then
		echo "allow=$allow" >> $cfile
	fi
	if [ "$session" != "" ]; then
		echo "session=$session" >> $cfile
	else
		echo "session=1" >> $cfile
	fi

	ufile=$config_dir/miniserv.users
	if [ "$crypt" != "" ]; then
		echo "$login:$crypt:0" > $ufile
	else
		$perl -e 'print "$ARGV[0]:",crypt($ARGV[1], "XX"),":0\n"' "$login" "$password" > $ufile
	fi
	chmod 600 $ufile
	echo "userfile=$ufile" >> $cfile

	kfile=$config_dir/miniserv.pem
	cp $wadir/miniserv.pem $kfile
	chmod 600 $kfile
	echo "keyfile=$config_dir/miniserv.pem" >> $cfile

	chmod 600 $cfile
	echo "..done"
	echo ""

	echo "Creating access control file.."
	afile=$config_dir/webmin.acl
	rm -f $afile
	echo "$login: $allmods" >> $afile
	chmod 600 $afile
	echo "..done"
	echo ""

fi

if [ "$noperlpath" = "" ]; then
	echo "Inserting path to perl into scripts.."
	(find $wadir -name '*.cgi' -print ; find $wadir -name '*.pl' -print) | $perl $wadir/perlpath.pl $perl -
	echo "..done"
	echo ""
fi

echo "Creating start and stop scripts.."
rm -f $config_dir/stop $config_dir/start
echo "#!/bin/sh" >>$config_dir/start
echo "echo Starting Webmin server in $wadir" >>$config_dir/start
echo "trap '' 1" >>$config_dir/start
uname -a | grep -i 'HP/*UX' >/dev/null
if [ $? = "0" ]; then
	echo "exec $wadir/miniserv.pl $config_dir/miniserv.conf &" >>$config_dir/start
else
	echo "exec $wadir/miniserv.pl $config_dir/miniserv.conf" >>$config_dir/start
fi

echo "#!/bin/sh" >>$config_dir/stop
echo "echo Stopping Webmin server in $wadir" >>$config_dir/stop
echo "kill \`cat $var_dir/miniserv.pid\`" >>$config_dir/stop
chmod 755 $config_dir/start $config_dir/stop
echo "..done"
echo ""

if [ "$upgrading" = 1 ]; then
	echo "Updating config files.."
else
	echo "Copying config files.."
fi
$perl $wadir/copyconfig.pl $os_type $os_version $wadir $config_dir "" $allmods
if [ "$upgrading" != 1 ]; then
	echo "os_type=$os_type" >> $config_dir/config
	echo "os_version=$os_version" >> $config_dir/config
	echo "real_os_type=$real_os_type" >> $config_dir/config
	echo "real_os_version=$real_os_version" >> $config_dir/config
	if [ -r /etc/system.cnf ]; then
		# Found a caldera system config file .. get the language
		source /etc/system.cnf
		if [ "$CONF_LST_LANG" = "us" ]; then
			CONF_LST_LANG=en
		elif [ "$CONF_LST_LANG" = "uk" ]; then
			CONF_LST_LANG=en
		fi
		grep "lang=$CONF_LST_LANG," $wadir/lang_list.txt >/dev/null 2>&1
		if [ "$?" = 0 ]; then
			echo "lang=$CONF_LST_LANG" >> $config_dir/config
		fi
	fi
else
	# one-off hack to set log variable in config from miniserv.conf
	grep log= $config_dir/config >/dev/null
	if [ "$?" = "1" ]; then
		grep log= $config_dir/miniserv.conf >> $config_dir/config
		grep logtime= $config_dir/miniserv.conf >> $config_dir/config
		grep logclear= $config_dir/miniserv.conf >> $config_dir/config
	fi
fi
echo $ver > $config_dir/version
echo "..done"
echo ""

if [ "$makeboot" = "1" ]; then
	echo "Configuring Webmin to start at boot time.."
	(cd $wadir/init ; WEBMIN_CONFIG=$config_dir WEBMIN_VAR=$var_dir $wadir/init/atboot.pl)
	echo "..done"
	echo ""
fi

if [ "$nouninstall" = "" ]; then
	echo "Creating uninstall script $config_dir/uninstall.sh .."
	cat >$config_dir/uninstall.sh <<EOF
#!/bin/sh
printf "Are you sure you want to uninstall Webmin? (y/n) : "
read answer
printf "\n"
if [ "\$answer" = "y" ]; then
	$config_dir/stop
	if [ "$atboot" = "1" ]; then
		(cd $wadir/init ; WEBMIN_CONFIG=$config_dir WEBMIN_VAR=$var_dir $wadir/init/delboot.pl)
	fi
	echo "Deleting $wadir .."
	rm -rf $wadir
	echo "Deleting $config_dir .."
	rm -rf $config_dir
	echo "Done!"
fi
EOF
	chmod +x $config_dir/uninstall.sh
	echo "..done"
	echo ""
fi

echo "Changing ownership and permissions .."
chown -R root:bin $config_dir
chmod -R og-rw $config_dir
chmod 755 $config_dir/sendmail/config >/dev/null 2>&1
chmod 755 $config_dir/sendmail/autoreply.pl >/dev/null 2>&1
if [ "$nochown" = "" ]; then
	chown -R root:bin $wadir
	#chmod -R og-rwx $wadir
	if [ $var_dir != "/var" ]; then
		chown -R root:bin $var_dir
		chmod -R og-rwx $var_dir
	fi
fi
echo "..done"
echo ""

if [ "$nostart" = "" ]; then
	echo "Attempting to start Webmin mini web server.."
	$config_dir/start
	if [ $? != "0" ]; then
		echo "ERROR: Failed to start web server!"
		echo ""
		exit 14
	fi
	echo "..done"
	echo ""

	echo "***********************************************************************"
	echo "Webmin has been installed and started successfully. Use your web"
	echo "browser to go to"
	echo ""
	if [ "$ssl" = "1" ]; then
		echo "  https://$host:$port/"
	else
		echo "  http://$host:$port/"
	fi
	echo ""
	echo "and login with the name and password you entered previously."
	echo ""
	if [ "$ssl" = "1" ]; then
		echo "Because Webmin uses SSL for encryption only, the certificate"
		echo "it uses is not signed by one of the recognized CAs such as"
		echo "Verisign. When you first connect to the Webmin server, your"
		echo "browser will ask you if you want to accept the certificate"
		echo "presented, as it does not recognize the CA. Say yes."
		echo ""
	fi
fi

