﻿<?LassoScript

/* library.lasso

		This file is shared by each of the applications in the Examples Pack. It
	defines all the tags that the applications rely on to perform common tasks
	such as initializing required databases and groups. These tags are defined
	in the Application_ namespace. Each tag is preceded by a comment block
	which contains a description of the tag.

		This file should be included using the library
	<http://reference.omnipilot.com/LDMLReference.0.LassoApp?tag=673> tag.
	Because this file may introduce <script /> or <meta /> tags, this should be
	done between the calling document's <head></head> tags.

	The following tags are defined herein:
	
	Application_initialize
	Application_wasInitialized
*/

/*
	Application_initialize

	Application_initialize(<appname string>, <version>, <elements array>);

	The Application_initialize tag permits the application developer to group
all application specific initializations into a single activity. These
initializations can consist of databases or tables which need to be created,
groups that need to be created and the assignment of database and table
permissions to groups. This tag may require administrator privileges when it is 
used to initialize an application for th first time.

	Application_initialize accepts three parameters. The first parameter is the
name of the application which is being initialized. This name should uniquely
identify the application for the given Lasso site. The second parameter should 
indicate the version of the application that is being initialized. The third 
parameter should consist of an array of initialization elements which describe 
the types of initializations that should take place.

	If the tag is able to successfully perform each initialization element, then
it will record the application name and the version number into the 
lasso_application.initialize table. Before performing any 
initialization tasks, the tag will search that same database for the presence of
the combination of the application name and version number. If found, the tag 
will not perform any of the provided initialization elements and will return 
immediately.

	An individual initialization element consists of a keyword, followed by an
associated string or pair. The following keywords are recognized:

	-database 
	-table 
	-group 
	-sql

	Only the -database and -group keywords are permitted at the top level. The
-table keyword can only be used within the -database element, and the -group and
-sql keywords can be used in several contexts. What follows is a more complete
description of the usage of each keyword.

	-Sql 
		= <sql statement string> 
	The -sql keyword is used to identify a single sql statement that should be
executed depending on the state of its containing initialization element. For
example, the -sql keyword is used to supply the CREATE statement used to create
a new table or database.

	-Group 
		= pair(<group name string>=<group description string>) 
		OR 
		= pair(<group name string>=<db permissions string>) 
	The -group keyword is used to either create a new group or to assign a group
to a specific database or table, given a permissions set. When used outside of a
-database element, the -group keyword will create a new group. When used within
a -database or -table element, the -group keyword will assign permissions for
the element and group in question. All initial -group creation elements are
processed before any -database elements.

	Before performing any group creation or assignment actions, Lasso will check
to see if records for those elements already exist. If so, Lasso will skip the
action and leave the existing records unmodified.

	-Database 
		= pair(<database name> = 
			array([-table=... | -group=... | -sql=...]...)) 
	The -database keyword requires an associated pair. The first element of this
pair is the name of the database in question. The second element of this pair is
an array of additional initialization elements which will be performed on or 
within the given database's context.

	If the specified database does not currently exist, meaning that there is no
record of the database within Lasso's
lasso_internal.security_datasource_databases table, then all -sql keywords
directly located within that database's elements array will be executed in the
order they are encountered. If there are no -sql statements in the database's
elements array, then the tag will assume that the database is a SQLite database
and will call the Sqlite_CreateDatabase tag with the specified database name.
Lasso's internal database information cache is then flushed, resulting in the
new database's information being entered into the proper lasso_internal tables.
The database will then be used, by itself, in an inline. If an error occurs
doing this, then the Application_initialize tag will fail, presenting the
error_code and error_msg generated by the failed inline.

	Once a database is known to exist in the lasso_internal database, each
-group assignment is performed, if necessary. After that, each -table element is
analyzed.

	-Table 
		= pair(<table name> = 
			array([-group=... | -sql=...]...)) 
	The -table keyword requires an associated pair. The first element of this 
pair is the name of the table in question. The second element of this pair is an
array of additional initialization elements which will be performed on the given
table. A -table element can only exist inside of a -database element.

	The initialization of a table proceeds in the same manner as it does for a
database. If the table does not exist, each -sql is executed, using the
surrounding -database element as a context.

	Once the table exists, each -group assignment is performed.

	-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

	Example

	This example initializes one application specific database consisting of two
tables. It also creates a group and assigns database and table permissions to
it.

	Application_initialize('My Example App', array(
		-group = pair('My App Users'='The users for this application.'),
		-database = pair('myappdb'= array(
			-sql = 'CREATE DATABASE myappdb',
			-group = pair('My App Users'='I,S,A,U,D,X'),
			-table = pair('myfirsttable'= array(
				-sql = 'CREATE TABLE myfirsttable (id INTEGER)',
				-group = pair('My App Users'='I,S,A,U,D,X')
				)),
			-table = pair('mysecondtable'= array(
				-sql = 'CREATE TABLE mysecondtable (id INTEGER)',
				-group = pair('My App Users'='I,S,A,D')
				))
			))
		));

	-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

	Gotchas

	Bootstrap database - CREATE DATABASE X statements can only be executed in an
inline which contains an already valid -database. Passing in a -bootstrap 
keyword inside of the -database element will cause the create database statement
to be execute within the context of the database indicated by the -bootstrap.

	If no -sql statements are given in a -database element, sqlite_createdb will
be called in lieu of any -sql.

	File permissions are hard to support since the paths have to be created by
the server admin in the ServerAdmin app. 
*/

if: !(lasso_tagexists: 'application_initialize');
	define_tag('initialize', -namespace='Application_', -req='appName', -type='string', -req='version', -req='config', -type='array');
		local('ret'=array); // an array of pairs where ->first is the error code and ->second is the error name
		if (!Application_wasInitialized(#appName, #version));
			auth_admin;
			
			inline(-database='lasso_application', -table='initialized');
				if (error_code == error_code_connectioninvalid || error_code == error_code_datasourceerror);
					sqlite_createdb('lasso_application');
					admin_reloaddatasource(1);
					inline(-database='lasso_internal', -sql='UPDATE security_datasource_databases SET enabled=\'Y\' WHERE name=\'lasso_application\'');/inline;
					admin_reloaddatasource(1);
					inline(-database='lasso_application', -sql='CREATE TABLE initialized (id INTEGER PRIMARY KEY, name TEXT, version TEXT)');
						inline(-sql='CREATE INDEX name_index ON initialized (name, version)');
						/inline;
					/inline;
					inline(-database='lasso_application', -table='initialized');/inline;
					// add perms to search this db
					inline(-database='lasso_internal', -sql="SELECT tables.id AS id_table, databases.id AS id_database 
															FROM 
															security_datasource_databases AS databases
															LEFT JOIN security_database_tables AS tables
															ON
															tables.id_database = databases.id
															WHERE 
															databases.name = 'lasso_application'");
						inline(-sql='INSERT INTO security_group_table_map (id_table, id_group, allow) 
										VALUES ( 'field('id_table')', 1, \'I,S\')');
						/inline;
						inline(-sql='INSERT INTO security_group_db_map (id_database, id_group, allow) 
										VALUES ( 'field('id_database')', 1, \'I,S\')');
						/inline;
					/inline;
				/if;
			/inline;
			
			local('groups' = #config->find(-group));
			local('databases' = #config->find(-database));
			local('bootStrap' = #config->find(-bootstrap));
			#bootStrap->size? 
				#bootStrap = #bootStrap->first->second | 
					#bootStrap = 'lasso_internal';
			// GROUPS
			inline(-database='lasso_internal', -table='security_groups');
				
				local('groupCreates'=array);
				
				iterate(#groups, local('curr'));
					// these are group creation elements
					local('gName', 'gDesc');
					if (#curr->second->isa('string'));
						#gName = #curr->second;
					else;
						protect;
							#gName = #curr->second->first;
							#gDesc = #curr->second->second;
						/protect;
					/if;
					fail_if(#gName->size == 0, error_code_invalidparameter, 'Empty group name on -group 'loop_count': '#curr);
					// see if it exists
					inline(-search, -op='eq', 'name'= #gName);
						if (found_count == 0);
							#groupCreates->insert(#gName=#gDesc); // it doesn't
						/if;
					/inline;
				/iterate;
				
				iterate(#groupCreates, local('curr'));
					inline(-add, 'created'=date->format('%Q %T'), 'id_site'=site_id, 'name'=#curr->first, 'enabled'='Y', 'comment'=#curr->second);
						#ret->insert(error_code='Group '#curr->first' created with error_code: 'error_code' error_msg: 'error_msg);
						log_detail(#ret->last->second);
					/inline;
				/iterate;
				admin_refreshsecurity;
			/inline;
			
			// DATABASES
			
			iterate(#databases, local('curr'));
				local('info' = #curr->second);
				// info is pair
				// info->first is db name
				// info->second is array
				local('dbName' = #info->first);
				local('dbId'= -1);
				
				// does it exist already?
				inline(-database='lasso_internal', -table='security_datasource_databases', -op='eq', 'name'=#dbName, -returnField='id', -search);
					if (found_count == 0); // no, it does not
						// execute all the -sql
						local('sqls' = #info->second->find(-sql));
						if (#sqls->size);
							inline(-database=#bootStrap);
								iterate(#sqls, local('currSql'));
									inline(-sql=#currSql->second);
										#ret->insert(error_code='Executed sql "'#currSql->second'" using bootstrap database '#bootStrap' with error_code: 'error_code' error_msg: 'error_msg);
										log_detail(#ret->last->second);
									/inline;
								/iterate;
							/inline;
						else;
							sqlite_createdb(#dbName);
							#ret->insert(error_code='Called sqlite_createdb('#dbName')');
							log_detail(#ret->last->second);
							admin_reloaddatasource(1);
						/if;
						// does it exist now?
						inline(-database='lasso_internal', -table='security_datasource_databases', -op='eq', 'name'=#dbName, -returnField='id', -search);
							if (found_count == 0); // no, it does not
								#ret->insert(error_code='Database '#dbName' still does not exist. Check the -sql statements for this database.');
								log_detail(#ret->last->second);
								// do nothing more on this -database
							else;
								#dbId = field('id');
								// enable it
								inline(-update, -keyfield='id', -keyvalue=#dbId, 'enabled'='Y');/inline;
								admin_reloaddatasource(1);
							/if;
						/inline;
					else;
						#dbId = field('id');
					/if;
				/inline;
				if (#dbId == -1);
					loop_continue;
				/if;
				// the database exists
				// GROUP ASSIGNMENTS
				local('groups' = #info->second->find(-group));
				iterate(#groups, local('group'));
					local('groupName' = #group->second->first);
					local('groupPrivs' = #group->second->second);
					local('groupId' = -1);
					// find the group id
					inline(-database='lasso_internal', -table='security_groups', -op='eq', 'name'=#groupName, -returnField='id', -search);
						fail_if (found_count == 0,  
							error_code_invalidparameter, 'The group '#groupName' does not exist.');
						#groupId = field('id');
					/inline;
					
					// see if these is already a mapping for this group/db combo
					inline(-database='lasso_internal', -table='security_group_db_map', 
							-op='eq', 'id_database'=#dbId, -op='eq', 'id_group'=#groupId, -search);
						if (found_count == 0);
							inline('id_database'=#dbId, 'id_group'=#groupId, 'allow'=#groupPrivs, -add);
								#ret->insert(error_code='Permissions assignment of '#groupPrivs' added for database '#dbName' and group '#groupName' with error_code: 'error_code' error_msg: 'error_msg);
								log_detail(#ret->last->second);
							/inline;
						/if;
					/inline;
				/iterate;
				
				// TABLES	
				local('tables' = #info->second->find(-table));
				iterate(#tables, local('table'));
					local('tableName' = #table->second->first);
					local('tableId' = -1);
					local('elements' = #table->second->second);
					
					// does the table exist?
					inline(-database='lasso_internal', -table='security_database_tables',
							-op='eq', 'id_database'=#dbId, -op='eq', 'name'=#tableName, -returnField='id', -search);
						if (found_count == 0);
							// no, it does not exist
							#ret->insert(error_code='Table '#tableName' was not found to exist.');
							log_detail(#ret->last->second);
							local('sqls' = #elements->find(-sql));
							if (#sqls->size);
								inline(-database=#dbName);
									iterate(#sqls, local('currSql'));
										inline(-sql=#currSql->second);
											#ret->insert(error_code='Executed sql "'#currSql->second'" using database '#dbName' with error_code: 'error_code' error_msg: 'error_msg);
											log_detail(#ret->last->second);
										/inline;
									/iterate;
								/inline;
							/if;
							admin_reloaddatasource(1);
							inline(-database=#dbName);/inline;
							// check it again
							inline(-database='lasso_internal', -table='security_database_tables',
									-op='eq', 'id_database'=#dbId, -op='eq', 'name'=#tableName, -returnField='id', -search);
								if (found_count == 0);
									#ret->insert(error_code='Table '#tableName' still does not exist. Check the -sql statements for this table.');
									log_detail(#ret->last->second);
									// do nothing more on this -table
								else;
									#tableId = field('id');
									// update the encoding to be UTF-8
									inline(-update, 'encoding'='UTF-8', -keyfield='id', -keyvalue=#tableId);/inline;
								/if;
							/inline;
						else;
							// yes it exists
							#tableId = field('id');
						/if;
					/inline;
					if (#tableID == -1);
						loop_continue;
					/if;
					// the table exists
					
					// GROUP ASSIGNMENTS
					
					local('groups' = #elements->find(-group));
					iterate(#groups, local('group'));
						local('groupName' = #group->second->first);
						local('groupPrivs' = #group->second->second);
						local('groupId' = -1);
						// find the group id
						inline(-database='lasso_internal', -table='security_groups', -op='eq', 'name'=#groupName, -returnField='id', -search);
							fail_if (found_count == 0,  
								error_code_invalidparameter, 'The group '#groupName' does not exist.');
							#groupId = field('id');
						/inline;
						
						// see if these is already a mapping for this group/table combo
						inline(-database='lasso_internal', -table='security_group_table_map', 
								-op='eq', 'id_table'=#tableId, -op='eq', 'id_group'=#groupId, -search);
							if (found_count == 0);
								inline('id_table'=#tableId, 'id_group'=#groupId, 'allow'=#groupPrivs, -add);
									#ret->insert(error_code='Permissions assignment of '#groupPrivs' added for table '#tableName' and group '#groupName' with error_code: 'error_code' error_msg: 'error_msg);
									log_detail(#ret->last->second);
								/inline;
							/if;
						/inline;
					/iterate;				
				/iterate;
			/iterate;
			// commit this to the database
			// if it completed successfully
			if (!local_defined('noCommit'));
				local('zeros' = #ret->find(0));
				if (#zeros->size == #ret->size); // no errors
					inline(-database='lasso_application', -table='initialized', -add,
							'name' = #appName, 'version'=#version);
						#ret = null;
					/inline;
				/if;
			/if;
			global('Application_initialize_'#appName) = string(date);
		/if;
		handle_error;
			log_detail(#ret);
		/handle_error;
		return: #ret;
	/define_tag;
/if;
/*
	Application_wasInitialized
	
	Application_wasInitialized(<appname string>, <version>)
	
	This tag returns true if the application, identified by a unique name and 
version number, has already been successfully initialized via the 
Application_initialize tag. Although it is safe to call Application_initialize 
multiple times for the same application, some applications may wish to avoid 
doing that based on the result of this tag.

*/
if: !(lasso_tagexists: 'application_wasinitialized');
	define_tag('wasInitialized', -namespace='Application_', -req='appName', -type='string', -req='version');
		global('Application_initialize_'#appName) != null ?
			return: true;
		// search in the database
		inline(-database='lasso_application', -table='initialized', -search,
				-op='eq', 'name' = #appName, -op='eq', 'version'=#version);
			found_count != 0 ?
				return: true;
		/inline;
		return: false;
	/define_tag;
/if;

/*
	Lasso Professional 8.0.x compatibility.
	The following code defines the tag [Site_ID] if it does not exist.
	This allows the example to work prior to LP 8.1 without modifying
	the code above.
*/

if: !(lasso_tagexists: 'site_id');
	define_tag: 'id', -namespace='site'; return: lasso_siteid; /define_tag;
/if;

?>